/***************************************************************** 
*  @file CRYS_HOST_TST_DH.c
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/



/********************** Include Files *************************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_qa_op_code.h"
#include "gen.h"
#include "error.h"
#include "CRYS_HOST_TST_DH.h"
#include "ExtApp_CRYS_Defines.h"


/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/




/************************ Public Functions ******************************/

 
/***********************************************************************
* Function Name:                                                           
*  SEPQA_DH_PKCS3         
* 
* Inputs:
* Generator_ptr 		- Pointer to the Generator octet string
* GeneratorSize 		- Size of the Generator String (in bytes)
* Prime_ptr 			- Pointer to the Prime octet string
* PrimeSize 			- Size of the Prime string (in bytes)
* L 					- Relevant only for PKCS#3, the private value length in bits
*	If L != 0 then - force the private key to be 2^(L-1) <= Prv < 2^l
*	if L 0 then: 0< Prv < P-1
*	In ANSI X9.42 L is irrelevant  
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Diffie-Hellman key agreement 
*  scheme according to PKCS#3  
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************************/                                
EXPORT_C CRYSError_t SEPQA_DH_PKCS3(
                           DxUint8_t   *Generator_ptr,
						   DxUint16_t   GeneratorSize,
						   DxUint8_t   *Prime_ptr,
						   DxUint16_t   PrimeSize,
						   DxUint16_t   L,
						   DxUint8_t   *FailureFunctionNameOut_ptr,
                           DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_ECC_DH_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_ECC_DH_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_DH_PKCS3;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_DH_GENERATORSIZE_IN_OFFSET] = GeneratorSize;
    Host_messageParam[DX_SEP_HOST_DH_PRIMESIZE_IN_OFFSET] = PrimeSize;
    Host_messageParam[DX_SEP_HOST_DH_L_IN_OFFSET] = L;
    
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send generator buffer */
   Host_maxLength = CRYS_DH_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Generator_ptr , 
                                     GeneratorSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send prime buffer */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Prime_ptr , 
                                     PrimeSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_DH_PKCS3 */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_DH_ANSI942         
* 
* Inputs:
* Generator_ptr 		- Pointer to the Generator octet string
* GeneratorSize 		- Size of the Generator String (in bytes)
* Prime_ptr 			- Pointer to the Prime octet string
* PrimeSize 			- Size of the Prime string (in bytes)
* Q_ptr 				- Relevant only for ANSI X9.42 - Pointer to the Q octet string
*							1<= Prv <= q-1   or   1< Prv < q-1
* QSize	 			    - Relevant only for ANSI X9.42 - Size of the Q string (in bytes)  				 				  
* SecretDataGenType     - secret data gen type
* KeyGetSecretKeyType   - get secret key type
* KeyLenInBits          - key len in bits
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes Diffie-Hellman key agreement 
*  scheme according to ANSI942  
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************************/                                
EXPORT_C CRYSError_t SEPQA_DH_ANSI942(
                           DxUint8_t   *Generator_ptr,
						   DxUint16_t   GeneratorSize,
						   DxUint8_t   *Prime_ptr,
						   DxUint16_t   PrimeSize,
						   DxUint8_t   *Q_ptr,
						   DxUint16_t   QSize,
						   DxUint16_t   SecretDataGenType,
						   DxUint16_t   KeyGetSecretKeyType,
						   DxUint16_t	KeyLenInBits,
						   DxUint8_t   *FailureFunctionNameOut_ptr,
                           DxUint32_t   FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = DX_SEP_HOST_SEP_ECC_DH_MSG_LENGTH_IN_WORDS;

   /* parameters */
   DxUint32_t   Host_messageParam[DX_SEP_HOST_SEP_ECC_DH_MSG_LENGTH_IN_WORDS] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_DH_ANSI942;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
    Host_messageParam[DX_SEP_HOST_DH_GENERATORSIZE_IN_OFFSET] = GeneratorSize;
    Host_messageParam[DX_SEP_HOST_DH_PRIMESIZE_IN_OFFSET] = PrimeSize;
    Host_messageParam[DX_SEP_HOST_DH_QSIZE_IN_OFFSET] = QSize;
    Host_messageParam[DX_SEP_HOST_DH_SecretDataGenType_IN_OFFSET] = SecretDataGenType;
    Host_messageParam[DX_SEP_HOST_DH_KeyGetSecretKeyType_IN_OFFSET] = KeyGetSecretKeyType;
    Host_messageParam[DX_SEP_HOST_DH_KeyLenInBits_IN_OFFSET] = KeyLenInBits;
    
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send generator buffer */
   Host_maxLength = CRYS_DH_MAX_VALID_KEY_SIZE_VALUE_IN_BITS/8;
   
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Generator_ptr , 
                                     GeneratorSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send prime buffer */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Prime_ptr , 
                                     PrimeSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send Q buffer */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Q_ptr , 
                                     QSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_DH_PKCS3 */







